import numpy as np

def create_alias_table(area_ratio):
    """
    从一个离散概率分布中创建alias表。

    :param area_ratio: 概率列表，总和应为1。
    :return: accept, alias. `accept[i]` 是接受索引 `i` 的概率，
             `alias[i]` 是当不接受 `i` 时备选的另一个索引。
    """
    l = len(area_ratio)
    accept, alias = [0] * l, [0] * l
    small, large = [], []
    # 将概率乘以N，使平均概率为1
    area_ratio_ = np.array(area_ratio) * l
    # 将索引分为两组：概率小于1的（small）和大于等于1的（large）
    for i, prob in enumerate(area_ratio_):
        if prob < 1.0:
            small.append(i)
        else:
            large.append(i)

    # 贪心地用大概率项填充小概率项的剩余空间
    while small and large:
        small_idx, large_idx = small.pop(), large.pop()
        accept[small_idx] = area_ratio_[small_idx]
        alias[small_idx] = large_idx
        # 更新大概率项的概率
        area_ratio_[large_idx] = area_ratio_[large_idx] - \
                                 (1 - area_ratio_[small_idx])
        if area_ratio_[large_idx] < 1.0:
            small.append(large_idx)
        else:
            large.append(large_idx)

    # 剩余的项，其接受概率都设为1
    while large:
        large_idx = large.pop()
        accept[large_idx] = 1
    while small:
        small_idx = small.pop()
        accept[small_idx] = 1

    return accept, alias


def alias_sample(accept, alias):
    """
    使用alias表进行一次采样。

    :param accept: accept表
    :param alias: alias表
    :return: 采样得到的索引
    """
    N = len(accept)
    # 随机选择一列（一个桶）
    i = int(np.random.random() * N)
    # 生成一个随机数，决定是选择主事件还是备选事件
    r = np.random.random()
    if r < accept[i]:
        # 接受，返回当前索引
        return i
    else:
        # 拒绝，返回alias表中对应的备选索引
        return alias[i]
